{
  lib,
  config,
  pkgs,
  osConfig,
  ...
}:

let
  cfg = config.home.ltp.hyprland;

  inherit (lib)
    mkEnableOption
    types
    mkOption
    mkIf
    lists
    ;

  workspaces = [
    "1"
    "2"
    "3"
    "4"
    "5"
    "6"
    "7"
    "8"
    "9"
  ];
  directions = [
    "left"
    "down"
    "up"
    "right"
  ];
in
{
  imports = [
    ./cliphist
    ./dunst
    ./hyprpaper
    ./rofi
    ./waybar
  ];

  options.home.ltp.hyprland = {
    enable = mkEnableOption {
      default = false;
      type = types.bool;
    };

    cursors = {
      package = mkOption {
        default = null;
        type = types.package;
      };

      name = mkOption {
        default = null;
        type = types.str;
      };

      size = mkOption {
        default = 24;
        type = types.int;
      };
    };
  };

  config = mkIf cfg.enable {
    # Service for brightness and volume indication, also provides lightctl + volumectl
    services.avizo = {
      enable = true;
      settings = {
        default = {
          # needs adjusting
          background = "rgba(69,71,90, 0.8)";
          border-color = "rgba(69, 71, 90, 0.8)";
          border-radius = 10;
        };
      };
    };

    home.packages = builtins.attrValues { inherit (pkgs) nautilus alacritty tela-circle-icon-theme; };

    home.pointerCursor = {
      inherit (cfg.cursors) name;
      inherit (cfg.cursors) package;
      inherit (cfg.cursors) size;
      gtk.enable = true;
      x11.enable = true;
      x11.defaultCursor = cfg.cursors.name;
    };

    gtk = {
      enable = true;
      theme = {
        package = pkgs.orchis-theme;
        name = "Orchis-Grey-Dark";
      };
      iconTheme = {
        package = pkgs.tela-icon-theme;
        name = "Tela-blue-dark";
      };
      cursorTheme = {
        inherit (cfg.cursors) package;
        inherit (cfg.cursors) name;
      };
    };

    wayland.windowManager.hyprland = {
      enable = true;

      plugins = builtins.attrValues { inherit (pkgs.hyprlandPlugins) hy3 hyprspace; };

      settings = {
        "$mod" = "SUPER";
        "$terminal" = "alacritty";
        "$fileManager" = "nautilus";

        env = [
          "XCURSOR_SIZE, 24"
          "NIXOS_OZONE_WL, 1"
        ];

        exec-once =
          let
            execlist = [
              "hyprctl setcursor ${cfg.cursors.name} ${builtins.toString cfg.cursors.size}"
              "waybar"
              "nm-applet"
              (mkIf osConfig.ltp.network.tailscale.enable "tailscale-systray")
            ];
          in
          execlist;

        general = {
          layout = "hy3";
          gaps_in = 5;
          gaps_out = 13;
          border_size = 0;
        };

        decoration = {
          rounding = 10;
          inactive_opacity = 0.8;
          dim_inactive = true;

          #drop shadow settings
          drop_shadow = true;
          shadow_range = 20;
          shadow_render_power = 3;
          shadow_ignore_window = true;
          "col.shadow" = "rgba(00000045)";
        };

        hy3 =
          {
          };

        misc = {
          force_default_wallpaper = 0;
          disable_hyprland_logo = false;
        };

        input = {
          kb_layout = "gb";

          touchpad = {
            natural_scroll = false;
          };
        };

        layerrule = [
          "dimaround, rofi"
          "blur, rofi"
          "blur, waybar"
          "ignorealpha 0, waybar"
        ];

        windowrulev2 = [
          "float, title:(Welcome to JetBrains Rider)"
        ];

        bind =
          let
            movewindow = lists.forEach directions (i: "SUPER_SHIFT, ${i}, hy3:movewindow, ${i}");
            movefocus = lists.forEach directions (i: "$mod, ${i}, hy3:movefocus, ${i}");
            movetoworkspace = lists.forEach workspaces (i: "SUPER_SHIFT, ${i}, movetoworkspace, ${i}");
            switchtoworkspace = lists.forEach workspaces (i: "$mod, ${i}, workspace, ${i}");
            controlbinds = [
              ", XF86AudioRaiseVolume, exec, volumectl -u up"
              ", XF86AudioLowerVolume, exec, volumectl -u down"
              ", XF86AudioMute, exec, volumectl toggle-mute"
              ", XF86MonBrightnessUp, exec, lightctl up"
              ", XF86MonBrightnessDown, exec, lightctl down"
            ];
            generalbinds = [
              "$mod, RETURN, exec, $terminal"
              "$mod, E, exec, $fileManager -w"
              #"$mod, V, hy3:makegroup, v, force_ephemeral"
              "SUPER_SHIFT, Q, hy3:killactive"
              "$mod, F, fullscreen"
              "SUPER_SHIFT, F, fullscreen, 1"
            ];
            hyprspacebinds = [
              "$mod, O, overview:toggle"
            ];
          in
          movewindow
          ++ movefocus
          ++ movetoworkspace
          ++ switchtoworkspace
          ++ controlbinds
          ++ generalbinds
          ++ hyprspacebinds;
      };

      extraConfig = ''

        bind = $mod, R, submap, resize

        # will start a submap called "resize"
        submap = resize

        # sets repeatable binds for resizing the active window
        binde = , right, resizeactive, 10 0
        binde = , left, resizeactive, -10 0
        binde = , up, resizeactive, 0 -10
        binde = , down, resizeactive, 0 10

        # use reset to go back to the global submap
        bind = , escape, submap, reset

        # will reset the submap, which will return to the global submap
        submap = reset
      '';
    };
  };
}
